function [K,sigma,x,xk] = parse(f,xd)
% parse  Parse a cell array of strings into matrices that represent a GMA.
%
% Usage
%
%   [K,sigma,x,xk] = parse(f,xd)
%
% Parameters
%
%   f: cell column vector of strings, system of functions in GMA form
%
%   xd: cell column vector of strings, dependent variables of the system
%
% Returns
%
%   K: any class 3-D array, kinetic orders of all terms in a GMA system
%
%   sigma: double matrix, numeric coefficents of all terms in a GMA system
%
%   x: cell column vector of strings, dependent and independent variables
%      (including rate constants) of the system; first N variables are the
%      dependent variables
%
%   xk: cell column vector of strings, variables in the kinetic orders

import dspace.Expression

m = length(f);
p = length(xd);

sigma = zeros(m,0);  % combination of alpha,beta
K = Expression(zeros(m,0,p));  % combination of g,h
x = xd;  % initialize x with dependent variables xd

f = Expression(f);  % parse the equations into an Expression array

for i = 1:m
    terms = split(f(i),Expression.OP_PLUS);
    n = length(terms);
    sigma(i,1:n) = ones(1,n);  % default coefficient (1)
    K(i,1:n,1:length(x)) = Expression(zeros(1,n,length(x)));
    
    for j = 1:n
        subterms = split(terms(j),Expression.OP_TIMES);
        
        for h = 1:length(subterms)
            sth = subterms(h);
            
            % Split subterm h into base and exponent.
            b = sth;
            e = Expression.ONE;  % default exponent
            if sth.op == Expression.OP_POWER
                b = sth.vals(1);
                e = sth.vals(2);
            end
            
            % Test for number or label and populate s,K accordingly.
            if isempty(labels(sth))
                sigma(i,j) = sigma(i,j) * double(sth);
            elseif islabel(b)
                [tf,xi] = ismember(char(b),x);
                if tf && (K(i,j,xi) == Expression.ZERO)
                    K(i,j,xi) = e;
                elseif tf
                    K(i,j,xi) = K(i,j,xi) + e;
                else
                    x = [x;char(b)];
                    K(i,j,length(x)) = e;
                end
            else
                error(dspace.Settings.ERROR_INVALID_INPUT, ...
                    'Equations are not in standard GMA form.');
            end
        end
    end
end

K = dspace.util.safe_permute(K,[1,3,2]);  % put into standard form
xk = cellstr(labels(K));  % extract labels for convenience
if isempty(xk)
    K = double(K);
end

end

function terms = split(expr,op)
    terms = expr;    
    if expr.op == op
        terms = expr.vals;
    end
end

function f = islabel(expr)
    f = (expr.op == '.' && ischar(expr.vals));
end

        
