function [ymin,ymax] = linprog_bounds_nonstrict(A,b,ymin,ymax,yi)
% linprog_bounds_nonstrict  Bound the feasible polytope.
%
% Usage
%
%   [ymin,ymax] = linprog_bounds_nonstrict(A,b,ymin,ymax,yi)
%
% Parameters
%
%   A,b: double matrix and column vector, design space boundaries in standard
%        form, linear case (A*y + b > 0)
%
%   ymin,ymax: (optional) double column vectors, initial bounds on
%              parameters in logspace; default is [-Inf,Inf]
%
%   yi: (optional) integer column vector, indicies of parameters to bound;
%       default is all parameters
%
% Returns
%
%   ymin,ymax: double column vectors, updated bounds on parameters
%
% Notes
%
%   In the linear case, when the kinetic orders are constant, the boundaries
%   in design space are linear.  This function uses linear programming to
%   place bounds of the resulting polytope.

[m,n] = size(A);

if nargin < 3
    ymin = -inf(n,1);
end
if nargin < 4
    ymax = inf(n,1);
end
if nargin < 5
    yi = (1:n);
end

A = -A;  % switch sign to correspond to MATLAB linprog inequalities
obj = zeros(1,n);  % objective function
opt = optimset('Display', 'off');  % turn off display

for i = yi(:)'
    % Minimize yi.
    obj(i) = 1;
    [dummy,y,flag] = linprog(obj,A,b,[],[],ymin,ymax,[],opt);
    if flag == 1
        ymin(i) = y;
    elseif flag == -3
        ymin(i) = -Inf;
    else
        error('linprog did not converge');
    end

    % Maximize yi.
    obj(i) = -1;
    [dummy,y,flag] = linprog(obj,A,b,[],[],ymin,ymax,[],opt);
    if flag == 1
        ymax(i) = -y;
    elseif flag == -3
        ymax(i) = Inf;
    else
        error('linprog did not converge');
    end

    obj(i) = 0;  % return to initial state
end

end
