function f = linprog_feasible_nonstrict(A,b,ymin,ymax)
% linprog_feasible_nonstrict  Test for nonstrict feasibility.
%
% Usage
%
%   f = linprog_feasible_nonstrict(A,b,ymin,ymax)
%
% Parameters
%
%   A,b: double matrix and column vector, design space boundaries in standard
%        form, linear case (A*y + b > 0)
%
%   ymin,ymax: (optional) double column vectors, initial bounds on
%              parameters in logspace; default is [-Inf,Inf]
%
% Returns
%
%   f: boolean, true if valid
%
% Notes
%
%   In the linear case, when the kinetic orders are constant, the boundaries
%   in design space are linear.  This function uses linear programming to
%   test for validity of the boundaries, which is equivalent to searching
%   for a feasible region.
%
%   This function creates a linear programming problem using nonstrict
%   boundary inequalities (i.e. A*y + b >= 0).  As such, regions that have
%   width zero in any dimension, or regions that are simply a boundary,
%   will be deemed valid. The regions arise when "almost" mutually
%   exclusive boundaries exist (i.e. Ai*y + bi >= 0 & Ai*y + bi <= 0). Such
%   regions do not appear to be useful and are currently ignored in most
%   design space analysis.  To automatically exclude such regions, strict
%   inequalities should be used.
%   (see <a href="matlab: help dspace.linear.linprog_feasible_strict">linprog_feasible_strict</a>)

[m,n] = size(A);  % p = number of cases

if nargin < 3
    ymin = -inf(n,1);
    ymax = inf(n,1);
end

A = -A;  % switch sign to correspond to MATLAB linprog inequalities
lb = ymin;  % lower bounds
ub = ymax;  % upper bounds
obj = zeros(1,n);  % objective function = slack variable
opt = optimset('Display', 'off');  % turn off display

[dummy,fval,flag] = linprog(obj,A,b,[],[],lb,ub,[],opt);
f = (flag == 1);
    
end
