function f = linprog_feasible_strict(A,b,ymin,ymax)
% linprog_feasible_strict  Test for strict feasibility.
%
% Usage
%
%   f = linprog_feasible_strict(A,b,ymin,ymax)
%
% Parameters
%
%   A,b: double matrix and column vector, design space boundaries in standard
%        form, linear case (A*y + b > 0)
%
%   ymin,ymax: (optional) double column vectors, initial bounds on
%              parameters in logspace; default is [-Inf,Inf]
%
% Returns
%
%   f: boolean, true if valid
%
% Notes
%
%   In the linear case, when the kinetic orders are constant, the boundaries
%   in design space are linear.  This function uses linear programming to
%   test for validity of the boundaries, which is equivalent to searching
%   for a feasible region.
%   
%   Linear programming problems in standard form use nonstrict inequalities
%   (e.g. a >= b), as can be seen in MATLAB linprog.  To test for
%   validity using strict inequalities (e.g. a > b), slack variables must be
%   added.

[m,n] = size(A);  % p = number of cases

if nargin < 3
    ymin = -inf(n,1);
    ymax = inf(n,1);
end

% Add a slack variable, Ay + log(b) + z > 0; if min(z) < 0, then feasible. 
A = [A,ones(m,1)];  % add a slack variable
A = -A;  % switch sign to correspond to MATLAB linprog inequalities
lb = [ymin;-1];  % lower bounds; limit slack to slightly negative
ub = [ymax;Inf];  % upper bounds
obj = [zeros(1,n),1];  % objective function = slack variable
opt = optimset('Display', 'off');  % turn off display

[dummy,fval,flag] = linprog(obj,A,b,[],[],lb,ub,[],opt);
f = (flag == 1 & fval < -dspace.Settings.TOLERANCE);
    
end
