classdef Expression
% Expression  Class that serves as a base for a rudimentary symbolic computer
%             algebra system (CAS).
%
%   An Expression object is either a node (e.g. an operator like +, *, etc.)
%   or leaf (e.g. a number like 121 or label like X3) in an abstract syntax
%   tree (AST) that represents a mathematical equation.
%
%   The Expression class was created to:
%   
%       1. Preserve term order when parsing text-based GMA equations for the
%          dpsace package.
%       2. More closely mimic the double class and avoid the bugs and
%          inconsistencies of the MATLAB sym class.
%       3. Potentially serve as a replacement for MATLAB symbolic toolbox by
%          duplicating the limited set of symbolic operations required by the
%          dspace package.
%
%   Expression uses +, *, and -1 to represent - and /.  For example, 'a-b' is
%   is parsed as a+(b*(-1)) and 'a/b' is parsed as a*(b^(-1)).
%
%   Expression allows addition and multiplication of several terms at once.
%   So, 'a+b+c' is parsed as plus(a,b,c) rather than plus(a,plus(b,c)).
%
%   Expression does not put equations into a canonical form.  As a result,
%   term order is preserved, but expressions are not equal if they are
%   written differently, even if they are mathematically equal.
%
%   Expression does not attempt to simplify the results of operations (e.g.
%   [a,b]*[0,0]' = a*0+b*0).  As such, Expression is inefficient for larger
%   problems.
%
%   A "label" Expression represents a variable name and must begin with
%   letter and consist entirely of alphanumeric characters.
%
%   Static Methods (of interest):
%       <a href="matlab: help dspace.Expression.Expression">Expression</a>
%       <a href="matlab: help dspace.Expression.parse">parse</a>
%       <a href="matlab: help dspace.Expression.islabel">islabel</a>
%
%   Object Methods (of interest):
%       <a href="matlab: help dspace.Expression.gather">gather</a>
%       <a href="matlab: help dspace.Expression.evalat">evalat</a>
%       <a href="matlab: help dspace.Expression.inv">inv</a>
%       <a href="matlab: help dspace.Expression.labels">labels</a>
%       <a href="matlab: help dspace.Expression.unique">unique</a>
%       <a href="matlab: help dspace.Expression.cellstr">cellstr</a>
%       <a href="matlab: help dspace.Expression.sym">sym</a>

    properties
        vals = 0;  % single value or vector of number, label, or expressions
                   %   default is zero (similar to double)
        op = '.';  % code that represents the type of expression
                   %   default is OP_TERMINAL
    end
    
    properties (Constant)
        FORMAT_DISPLAY = '%0.5g';  % precision for displaying a number
        FORMAT_CONVERT = '%0.20g';  % precision for converting a number
                                    %  (e.g. from sym)
        
        REGEXP_NUMBER = '\d*\.?\d+(e(\+|-)?\d+)?';  % regexp for a number
        REGEXP_LABEL = '[a-zA-Z]\w*';  % regexp for a label
        
        % Common constants.
        ZERO = dspace.Expression(0,'.');
        ONE = dspace.Expression(1,'.');
        NEGONE = dspace.Expression(dspace.Expression.ONE,'m');
        
        % Op codes.
        OP_PLUS = '+';
        OP_MINUS = '-';
        OP_TIMES = '*';
        OP_DIVIDE = '/';
        OP_POWER = '^';
        OP_UPLUS = 'p';
        OP_UMINUS = 'm';
        OP_TERMINAL = '.';
        OP_UNDEFINED = '?';
        OP_ALL = '+-/*^mp.?';

        % Error codes.
        ERROR_INVALID_PARAMETER = 'ExpressionError:InvalidParameter';
        ERROR_MISMATCHED_DIMS = 'ExpressionError:MismatchedDims';
        ERROR_MALFORMED_EXPRESSION = 'ExpressionError:MalformedExpression';
        ERROR_MISSING_TOOLBOX = 'ExpressionError:MissingToolbox';
        ERROR_SYMBOLIC_TOOLBOX = 'ExpressionError:SymbolicToolbox';
    end
    
    methods
        function obj = Expression(vals,op)
        % Expression  Constructor for Expression class.
        %
        % Usage
        %
        %   obj = Expression(vals,op)
        %
        % Parameters
        %
        %   vals: vector of objects that can be converted to an Expression
        %
        %   op: (optional) code that represents the type of Expression;
        %       if not designated, the constructor attempts to convert vals
        %       to an appropriate Expression
        %
        % Returns
        %
        %   obj: Expression object or array
        %
        % Notes
        %
        %   Current op codes are:
        %
        %       Expression.OP_PLUS
        %       Expression.OP_MINUS
        %       Expression.OP_TIMES
        %       Expression.OP_DIVIDE
        %       Expression.OP_POWER
        %       Expression.OP_UPLUS
        %       Expression.OP_UMINUS
        %       Expression.OP_TERMINAL (i.e. number or label)
        
            % Handle two inputs, where op is designated.
            % WARN: still trusts input for -/^
            if nargin == 2
                if ~ismember(op, dspace.Expression.OP_ALL)
                    error(dspace.Expression.ERROR_INVALID_PARAMETER, ...
                        'op not recognized.');
                elseif (isnumeric(vals) && numel(vals) && ...
                       op == dspace.Expression.OP_TERMINAL) || ...  % number
                       (dspace.Expression.islabel(vals) && ...
                       op == dspace.Expression.OP_TERMINAL) || ... % label
                       (isa(vals,'dspace.Expression') && ...
                       op ~= dspace.Expression.OP_TERMINAL)
                    obj.vals = vals;
                    obj.op = op;
                else
                    error(dspace.Expression.ERROR_INVALID_PARAMETER, ...
                        'Cannot create Expression with given vals, op.');
                end
            
            % Handle one input, where the vals are converted as necessary.
            elseif nargin == 1
                if isempty(vals)
                    obj = dspace.Expression.empty(size(vals));
                elseif isnumeric(vals)
                    n = numel(vals);
                    obj(n,1) = dspace.Expression();  % allocates space
                    for i = 1:n
                        obj(i).vals = vals(i);
                        obj(i).op = dspace.Expression.OP_TERMINAL;
                    end
                    obj = reshape(obj,size(vals));
                elseif ischar(vals)
                    obj = dspace.Expression.parse(vals);
                elseif isa(vals,'dspace.Expression')
                    obj = vals;
                elseif iscell(vals)
                    n = numel(vals);
                    obj(n,1) = dspace.Expression();  % allocates space
                    for i = 1:n
                        obji = dspace.Expression(vals{i});
                        if numel(obji) == 1
                            obj(i) = obji;
                        else
                        	error(...
                                dspace.Expression.ERROR_INVALID_PARAMETER, ...
                                'Cannot handle a list of lists.');
                        end
                    end
                    obj = reshape(obj,size(vals));
                elseif isa(vals,'sym')
                    n = numel(vals);
                    obj(n,1) = dspace.Expression();  % allocates space
                    for i = 1:n
                        obj(i) = dspace.Expression(char(vals(i), ...
                            dspace.Expression.FORMAT_CONVERT));
                    end
                    obj = reshape(obj,size(vals));
                else
                 	error(dspace.Expression.ERROR_INVALID_PARAMETER, ...
                    	'Cannot convert vals to Expression.');
                end
            end
        end
    end
        
    methods (Static)
        function e = parse(s)
        % parse  Parse a string into an Expression.
        %
        % Usage
        %
        %   e = parse(s)
        %
        % Parameters
        %
        %   s: char array, represents a mathematical equation
        %
        % Returns
        %
        %   e: the resulting Expression

            % Replace all numbers and labels with references to an array of
            % Expression.
            re = [dspace.Expression.REGEXP_NUMBER,'|', ...
                  dspace.Expression.REGEXP_LABEL];
            [a,b] = regexp(s, re, 'match', 'split');
            z = a;
            for i = 1:length(z)
                if ~dspace.Expression.islabel(z{i})
                    z{i} = eval(z{i});  % convert string to a number
                end
                
                % Convert to terminal Expression.
                z{i} = dspace.Expression(z{i},dspace.Expression.OP_TERMINAL);
                
                % Put a reference to the Expression in the original equation.
                a{i} = ['z{',int2str(i),'}'];
            end

            % Reassemble the string (with the changes).
            s = [b;[a,{''}]];
            s = [s{:}];

            % Parse by highjacking the MATLAB parser.
            e = eval(s); 
        end
        
        function b = islabel(s)
        % islabel  Test whether a string is a qualified label.
        %
        % Usage
        %
        %   b = islabel(s)
        %
        % Parameters
        %
        %   s: char array
        %
        % Returns
        %
        %   b: boolean, true if s is a qualified label
        
            b = ischar(s) && ...
                strcmp(regexprep(s,dspace.Expression.REGEXP_LABEL,'a'),'a');
        end
    end
end

