function s = char(obj,format)
% char  Convert an Expression object to a string.
%
% Usage
%
%   s = char(obj,format)
%
% Parameters
%
%   obj: Expression object
%
%   format: (optional) string, indicates details of formating; default is
%           standard dispaly format for Expression
%           (see <a href="matlab: help sprintf">sprintf</a>)
%
% Returns
%
%   s: string
%
% Notes
%
%   char should produce a string that can be converted back to an
%   Expression via Expression.parse(s).

    import dspace.Expression

    if nargin < 2
        format = Expression.FORMAT_DISPLAY;
    end

    if numel(obj) ~= 1
        error(Expression.ERROR_INVALID_PARAMETER, ...
            'char only works for Expression arrays of size 1.');
    end

    if ismember(obj.op,[Expression.OP_PLUS, ...
                        Expression.OP_MINUS, ...
                        Expression.OP_TIMES, ...
                        Expression.OP_DIVIDE, ...
                        Expression.OP_POWER])
        vv = cell(1,length(obj.vals));
        for i = 1:length(vv)
            vv{i} = char(obj.vals(i));
        end
        s = strcat(obj.op,vv(2:end));  % prepend op to terms
        s = ['(',vv{1},s{:},')'];
    elseif obj.op == Expression.OP_UPLUS
        s = char(obj.vals);  % unary plus
    elseif obj.op == Expression.OP_UMINUS
        s = ['(-',char(obj.vals),')'];  % unary minus
    elseif obj.op == Expression.OP_TERMINAL && isnumeric(obj.vals)
        s = num2str(obj.vals,format);  % print terminal number
    elseif obj.op == Expression.OP_TERMINAL && ischar(obj.vals)   
        s = obj.vals;  % print terminal label
    else
        error(Expression.ERROR_MALFORMED_EXPRESSION, ...
            'Operator not recognized.');
    end
end
