function bnd = calculate_boundaries(kases)
% calculate_boundaries  Calculate design space boundaries.
%
% Usage
%
%   bnd = calculate_boundaries(kases)
%
% Parameters
%
%   kases: struct, design space cases in the standard form
%          (see <a href="matlab:help dspace.enumerate_cases">enumerate_cases</a>)
%
% Returns
%
%   bnd: struct, design space boundaries in standard form
%
%       bnd.W,.U,.zeta,.delta: any class 3-D array, any class 3-D array,
%                              double matrix, double matrix (respectively),
%                              boundaries in the form
%                              W*log(x) + U*log(zeta) + log(delta) > 0; each
%                              slice in dim 3 represents a different case
%
%       bnd.xi: cell column vector of strings, independent variables
%               (including rate constants) of the system
%
%       bnd.xk: cell column vector of strings, variables in the kinetic orders
%
%       bnd.sig: integer column vector, term signature in standard form
%                (see <a href="matlab:help dspace.gma.signature">signature</a>)
%
% Notes
%   
%   This function takes dominant S-systems and conditions, which are
%   functions of the dependent and indpendent variables, and calculates
%   boundaries in design space, which are functions of only the independent
%   variables (including rate constants and kinetic orders).
%
%   If the dependent kinetic order matrix (Ad) is rank deficient and
%   therefore not invertible, the resulting matrices are all NaN,
%   which is considered an invalid set of boundaries.

A = kases.G - kases.H;
[m,n,p] = size(A);
Ad = A(:,1:m,:);
Ai = A(:,m+1:n,:);
Cd = kases.C(:,1:m,:);
Ci = kases.C(:,m+1:n,:);

bnd.zeta = kases.alpha./kases.beta;
bnd.delta = kases.delta;
bnd.xi = kases.x(m+1:end);  % keep independent variables
bnd.xk = kases.xk;
bnd.sig = kases.sig;

% Initialize.
bnd.W = feval(str2func(class(Cd)),zeros(size(Cd)));  % make W same type as Cd
bnd.U = feval(str2func(class(Ci)),zeros(size(Ci)));  % make U same type as Ci

for i = 1:p
    
    % If Ad is rank deficient, make all W,U values NaN (assume invalid).
    if double(rank(Ad(:,:,i))) < m
        bnd.W(:,:,i) = feval(str2func(class(Cd)),nan(size(Cd(:,:,i))));
        bnd.U(:,:,i) = feval(str2func(class(Ci)),nan(size(Ci(:,:,i))));
        continue
    else
    
    bnd.W(:,:,i) = -Cd(:,:,i)*inv(Ad(:,:,i));
    bnd.U(:,:,i) = bnd.W(:,:,i)*Ai(:,:,i) + Ci(:,:,i);
end

end
