function all = combine_boundaries(bnd,kasenums)
% combine_boundaries  Combine sets of boundaries into a single set.
%
% Usage
%
%   all = combine_boundaries(bnd,kasenums)
%
% Parameters
%
%   bnd: struct, design space boundaries in standard form
%        (see <a href="matlab:help dspace.calculate_boundaries">calculate_boundaries</a>)
%
%   kasenums: (optional) integer column vector, indicies of desired sets;
%             default is all sets
%
% Returns
%
%   all: struct, design space boundaries in standard form, a single set
%        (see <a href="matlab:help dspace.calculate_boundaries">calculate_boundaries</a>)
%
% Notes
%
%   Generally, the struct of boundaries is intended to contain sets of
%   boundaries for every case, thereby representing all of design space.
%   After combining sets, or cases, the case numbers and term signature
%   lose their meaning.

import dspace.util.safe_permute

[m,n,p] = size(bnd.U);

if nargin < 2
    kasenums = (1:p)';
end

nk = length(kasenums);

% Combine the constraints for every case in bnd.
all.U = reshape(safe_permute(bnd.U(:,:,kasenums),[1,3,2]),m*nk,n);
all.W = dspace.util.blkdiag_3d(bnd.W(:,:,kasenums));  % W tiled on diag
all.zeta = reshape(bnd.zeta(:,:,kasenums),[],1);  % unroll to column vector
all.delta = reshape(bnd.delta(:,:,kasenums),[],1);  % unroll to column vector
all.xi = bnd.xi;
all.xk = bnd.xk;
all.sig = NaN;  % term signature loses meaning after combining boundaries

end