function box = create_boundingbox(bnd,pr,vars)
% create_boundingbox  Create a bounding box around a set of boundaries.
%
% Usage
%
%   box = create_boundingbox(bnd,pr,vars)
%
% Parameters
%
%   bnd: struct, design space boundaries in standard form
%        (see <a href="matlab:help dspace.calculate_boundaries">calculate_boundaries</a>)
%
%   pr: (optional) struct, additional parameter ranges, or constraints;
%       field names represent parameter names; min and max of field value(s)
%       indicate the range of the parameters
%
%   vars: (optional) cell array of strings, parameter names to be bounded;
%         default is all parameters
%
% Returns
%
%   box: struct column vector, box for each set of boundaries; field names
%        correspond to parameter names; field values are two element row
%        vectors with lower and upper bounds, respectively, in Cartesian
%        space; field order is not guaranteed
%
% Notes
%
%   The function creates an axis aligned bounding box (AABB), which is a
%   rough enclosure of the n-d area in designspace.  In the linear case, the
%   actual area is an n-d polytope and a more exact solution would be to
%   enumerate the faces or vertices of the polytope.

if nargin < 3 || isempty(vars)
    vars = unique(cat(1,bnd.xi,bnd.xk));
end
if nargin < 2 || isempty(pr)
    pr = struct();
end

% Test to see if the kinetic orders are pinned to a single value by pr.
keys = fieldnames(pr);
vals = struct2cell(pr);
i1 = (cellfun(@length,vals) == 1);
keys1 = keys(i1);
vals1 = vals(i1);

if isempty(bnd.xk)
    box = create_boundingbox_linear(bnd,pr,vars);    
elseif all(ismember(bnd.xk,keys1))
    bnd.U = double(dspace.util.safe_subs(bnd.U,keys1,vals1));
    bnd.W = double(dspace.util.safe_subs(bnd.W,keys1,vals1));
    box = create_boundingbox_linear(bnd,pr,vars);
else
    box = create_boundingbox_nonlinear(bnd,pr,vars);
end

end


function box = create_boundingbox_linear(bnd,pr,vars)

keys = fieldnames(pr);
vals = struct2cell(pr);

% By default, make the range of every param infinite.
n = length(bnd.xi);  % number of independent variables
ymin = -inf(n,1);
ymax = inf(n,1);

% Change the ranges of the variables according to pr.
[xb,xj] = ismember(bnd.xi,keys);
xvals = vals(xj(xb));
ymin(xb) = log(cellfun(@min,xvals));
ymax(xb) = log(cellfun(@max,xvals));

% Iterate over every case.
p = size(bnd.U,3);  % number of cases
ymin = repmat(ymin,1,p);
ymax = repmat(ymax,1,p);

% If not valid, min,max should be NaN,NaN.
ymin2 = nan(size(ymin));
ymax2 = nan(size(ymax));

% Determine which kases are valid.
v = dspace.is_valid(bnd,pr);
vn = find(v);  % kasenums of valid

for i = vn'
    A = bnd.U(:,:,i);
    b = bnd.W(:,:,i)*log(bnd.zeta(:,:,i)) + log(bnd.delta(:,:,i));
    [ymin2(:,i),ymax2(:,i)] = ...
        dspace.linear.linprog_bounds_nonstrict(A,b,ymin(:,i),ymax(:,i));
end

% Reform ymin,ymax to correspond to pr.
xmin = exp(ymin2);
xmax = exp(ymax2);
box = struct();
for i = 1:p
    for j = 1:n
        box(i).(bnd.xi{j}) = [xmin(j,i),xmax(j,i)];
    end
    % Add xk back in.
    for j = 1:length(bnd.xk)
        box(i).(bnd.xk{j}) = repmat(pr.(bnd.xk{j}),1,2);
    end
end

% Only return vars.
keys = fieldnames(box);
box = rmfield(box,keys(~ismember(keys,vars)));

end


function varargout = create_boundingbox_nonlinear(varargin)

error(dspace.Settings.ERROR_NOT_IMPLEMENTED, ...
    'Function create_boundingbox_nonlinear not implemented.');

end





   