function kases = enumerate_cases(gma)
% enumerate_cases  Enumerate dominant S-systems and conditions for dominance.
%
% Usage
%
%   kases = enumerate_cases(gma)
%
% Parameters
%
%   gma: struct, GMA in standard form
%        (see <a href="matlab:help dspace.parse_gma">parse_gma</a>)
%
% Returns
%
%   kases: struct, design space cases in standard form (see following fields)
%
%       kases.G,.H: any class 3-D array, kinetic orders of positive, negative
%                 terms (respectively) in dominant S-system; each slice in
%                 dim 3 represents a different case
%
%       kases.alpha,.beta: double matrix, numeric coefficients of positive,
%                          negative terms (respectively) in dominanty
%                          S-system system; only the absolute value of the
%                          coefficient is used; each slice in dim 3
%                          represents a different case
%   
%       kases.C,.delta: any class 3-D array, double matrix (respectively),
%                       conditions in the form C*log(x) + log(delta) > 0; each
%                       slice in dim 3 represents a different case
%       
%       kases.x: cell column vector of strings, dependent and independent
%                variables (including rate constants) of the system; first N
%                variables are the dependent variables
%
%       kases.xk: cell column vector of strings, variables in the kinetic
%                 orders
%
%       kases.sig: integer column vector, term signature in standard form
%                  (see <a href="matlab:help dspace.gma.signature">signature</a>)

import dspace.util.safe_permute

P = gma.sig(1:2:end);  % vector of num pos terms (i.e. [P1,P2,...])
N = gma.sig(2:2:end);  % vector of num neg terms (i.e. [P1,P2,...])
m = length(gma.sig)/2;  % num equations
n = length(gma.x);  % num vars
kases.x = gma.x;
kases.xk = gma.xk;
kases.sig = gma.sig;

s = prod(gma.sig);  % num cases
r = sum(gma.sig-1);  % num conditions
ci = cumsum(gma.sig-1);  % indicies of the conditions
ci = [0;ci(1:end-1)]+1;
pi = ci(1:2:end);
ni = ci(2:2:end);

% Initialize results.  Handle double, sym, or other.
kases.G = zeros(m,n,s);
kases.H = zeros(m,n,s);
kases.alpha = zeros(m,1,s);
kases.beta = zeros(m,1,s);
kases.C = zeros(r,n,s);
kases.delta = zeros(r,1,s);

klass = class(gma.G);
if ~strcmp(klass,class(gma.H))
    error(dspace.Settings.ERROR_INVALID_INPUT, ...
        'G and H are not the same class.');
end
kases.G = feval(str2func(klass),kases.G);  % cast kases.G to class of gma.G
kases.H = feval(str2func(klass),kases.H);  % cast kases.H to class of gma.G
kases.C = feval(str2func(klass),kases.C);  % cast kases.C to class of gma.G

for i = 1:s 
    tdom = dspace.dominant_terms(gma.sig,i);
    pdom = tdom(1:2:end);  % dominant indicies for pos terms
    ndom = tdom(2:2:end);  % dominant indicies for neg terms
    
    for j = 1:m  
        % Choose dominant terms for s-system.
        kases.alpha(j,:,i) = gma.alpha(j,pdom(j));
        kases.beta(j,:,i) = gma.beta(j,ndom(j));
        kases.G(j,:,i) = gma.G(j,:,pdom(j));
        kases.H(j,:,i) = gma.H(j,:,ndom(j));
        
        % Construct conditions.
        if P(j) > 1
            pdomj = repmat(pdom(j),P(j)-1,1);  % index of dominant term
            pnonj = (1:P(j));
            pnonj = pnonj(pnonj ~= pdom(j));  % indicies of nondominant terms

            kases.C(pi(j):pi(j)+P(j)-2,:,i) = ...
                safe_permute(gma.G(j,:,pdomj) - gma.G(j,:,pnonj),[3,2,1]);
            kases.delta(pi(j):pi(j)+P(j)-2,:,i) = ...
                safe_permute((gma.alpha(j,pdomj) ./ gma.alpha(j,pnonj)),[2,1]);
        end
        
        if N(j) > 1
            ndomj = repmat(ndom(j),N(j)-1,1);  % index of dominant term
            nnonj = (1:N(j));
            nnonj = nnonj(nnonj ~= ndom(j));  % indicies of nondominant terms

            kases.C(ni(j):ni(j)+N(j)-2,:,i) = ...
                safe_permute(gma.H(j,:,ndomj) - gma.H(j,:,nnonj),[3,2,1]);
            kases.delta(ni(j):ni(j)+N(j)-2,:,i) = ...
                safe_permute((gma.beta(j,ndomj) ./ gma.beta(j,nnonj)),[2,1]);
        end
    end
end

end


