function kaselist = find_intersections(bnd,pr,n,kasenums)
% find_intersections  Find all occurances of intersecting boundaries.
%
% Usage
%
%   kaselist = find_intersections(bnd,pr,n,kasenums)
%
% Parameters
%
%   bnd: struct, design space boundaries in standard form
%        (see <a href="matlab:help dspace.calculate_boundaries">calculate_boundaries</a>)
%
%   pr: (optional) struct, additional parameter ranges, or constraints;
%       field names represent parameter names; min and max of field value(s)
%       indicate the range of the parameters
%
%   n: (optional) integer column vector, number of intersecting cases; find
%      occurances of n intersecting cases, repeat for all n; default is to
%      check for all possible n (1 to length(kasenums))
%
%   kasenums: (optional) integer column vector, case numbers; default is
%             all cases
%
% Returns
%
%   kaselist: cell column vector, lists of intersecting cases; each entry is
%             an array of row vectors that indicate all occurances of n
%             n intersecting cases

p = size(bnd.U,3);  % num cases

if nargin < 4
    kasenums = (1:p)';  % check all kasenums
end
if nargin < 3 || isempty(n)
    n = (1:p);  % check for all possible numbers of intersections
end
if nargin < 2 || isempty(pr)
    pr = struct();
end

kaselist = cell(length(n),1);
n = sort(n);  % work from lowest number of intersections to highest
for j = 1:length(n)
    if isempty(kasenums)
        break;  % abort when kasenums is empty
    end
    
    % Create a list of all combinations (not permutations) of kasenums
    kk = nchoosek(kasenums(:)',n(j))';
    
    % For each combination of regions, check for intersection.
    f = false(size(kk,2),1);
    for i = 1:size(kk,2)
        f(i) = dspace.is_intersecting(bnd,pr,kk(:,i));
    end
    kaselist{j} = kk(:,f)';
    
    kasenums = unique(kaselist{j});  % only continue with intersecting kasenums
end

kaselist = kaselist(~cellfun(@isempty,kaselist));  % truncate empty elements

end
