function v = is_valid(bnd,pr)
% is_valid  Determine if boundaries enclose a valid area of design space.
%
% Usage
%
%   v = is_valid(bnd,pr)
%
% Parameters
%
%   bnd: struct, design space boundaries in standard form
%        (see <a href="matlab:help dspace.calculate_boundaries">calculate_boundaries</a>)
%
%   pr: (optional) struct, additional parameter ranges, or constraints;
%       field names represent parameter names; min and max of field value(s)
%       indicate the range of the parameters
%
% Returns
%
%   v: boolean column vector, true for each valid set of boundaries

if nargin < 2 || isempty(pr)
    pr = struct();
end

% Test to see if the kinetic orders are pinned to a single value by pr.
keys = fieldnames(pr);
vals = struct2cell(pr);
i1 = (cellfun(@length,vals) == 1);
keys1 = keys(i1);
vals1 = vals(i1);

if isempty(bnd.xk)
    v = is_valid_linear(bnd,pr);
elseif all(ismember(bnd.xk,keys1))
    bnd.U = double(dspace.util.safe_subs(bnd.U,keys1,vals1));
    bnd.W = double(dspace.util.safe_subs(bnd.W,keys1,vals1));
    v = is_valid_linear(bnd,pr);
else
    v = is_valid_nonlinear(bnd,pr);
end

end


function v = is_valid_linear(bnd,pr)

keys = fieldnames(pr);
vals = struct2cell(pr);

% By default, make the range of every param infinite.
n = length(bnd.xi);  % number of independent variables
ymin = -inf(n,1);
ymax = inf(n,1);

% Change the ranges of the variables according to pr.
[i,j] = ismember(bnd.xi,keys);
xvals = vals(j(i));
ymin(i) = log(cellfun(@min,xvals));
ymax(i) = log(cellfun(@max,xvals));

% Iterate over every set of boundaries.
p = size(bnd.U,3);  % number of cases
v = false(p,1);

for i = 1:p
    A = bnd.U(:,:,i);
    b = bnd.W(:,:,i)*log(bnd.zeta(:,:,i)) + log(bnd.delta(:,:,i));
    
    % If A is NaN, assume it is not valid.
    if (isa(A(1),'sym') && strcmp(char(A(1)),'NaN')) || isnan(A(1))
        v(i) = false;
        continue
    end
    
    v(i) = dspace.linear.linprog_feasible_strict(A,b,ymin,ymax);
end

end


function varargout = is_valid_nonlinear(varargin)

error(dspace.Settings.ERROR_NOT_IMPLEMENTED, ...
    'Function is_valid_nonlinear not implemented.');

end
