function [tol,kasenums] = measure_tolerance(bnd,p0,pr,vars,kasenums)
% measure_tolerance  Measure tolerance to boundaries.
%
% Usage
%
%   [tol,kasenums] = measure_tolerance(bnd,p0,pr,vars,kasenums)
%
% Parameters
%
%   bnd: struct, design space boundaries in standard form
%        (see <a href="matlab:help dspace.calculate_boundaries">calculate_boundaries</a>)
%
%   p0: struct, parameter values for the operating point; field names
%       represent parameter names; field values represent parameter values;
%       every parameter must be specified by a single value
%
%   pr: (optional) struct, additional parameter ranges, or constraints;
%       field names represent parameter names; min and max of field value(s)
%       indicate the range of the parameters
%
%   vars: (optional) cell array of strings, parameters names to be measured;
%         default is all parameters
%
%   kasenums: (optional) integer column vector, case numbers; default is
%             all cases
%
% Returns
%
%   tol: struct column vector, tolerance for each case; field names
%        correspond to parameter names; field values are two element row
%        vectors with fold change down and fold change up, respectively, in
%        Cartesian space; field order is not guaranteed
%
%   kasenums: integer column vector, case numbers in which the operating
%             point lies (and designated by input kasenums)

% =====
% Check input.

if nargin < 3 || isempty(pr)
    pr = struct();
end

keys = fieldnames(p0);
vals = struct2cell(p0);  % some param vals will be singular, some arrays

% Check that p0 pins all values.
[xb,xi] = ismember(bnd.xi,keys);
if ~all(xb)
    error(dspace.Settings.ERROR_MISSING_PARAMETERS, ...
          'Some parameters are missing.');
end

% Check that p0 is in pr.
k = intersect(keys,fieldnames(pr));
for i = 1:length(k)
    if length(pr.(k{i})) == 1 && pr.(k{i}) ~= p0.(k{i})
        error('some p0 not equal to pr');
    elseif length(pr.(k{i})) > 1 && ~(max(pr.(k{i})) > p0.(k{i}) && ...
                                      min(pr.(k{i})) < p0.(k{i}))
        error('some p0 not contained in pr');
    end
end

% Check that p0 has vars (if specified).
if nargin < 4 || isempty(vars)
    vars = keys;
elseif ~all(ismember(vars,keys))
    error('p0 does not contain specified vars');
end

% Find regions for which p0 is valid.  If p0 is valid, then pr is valid.
v = dspace.is_valid(bnd,p0);
vn = find(v);

% Check that p0 is in kasenums (if specified).
if nargin < 5
    kasenums = vn;
elseif sort(vn) ~= sort(kasenums)
    error('p0 not in specified cases');
end

% =====
% Calculate tolerance.

% Test to see if the kinetic orders are pinned to a single value by pr.
keys = fieldnames(pr);
vals = struct2cell(pr);
i1 = (cellfun(@length,vals) == 1);
keys1 = keys(i1);
vals1 = vals(i1);

if isempty(bnd.xk)
    [tol,kasenums] = measure_tolerance_linear(bnd,pr,p0,vars,kasenums);    
elseif all(ismember(bnd.xk,keys1))
    bnd.U = double(dspace.util.safe_subs(bnd.U,keys1,vals1));
    bnd.W = double(dspace.util.safe_subs(bnd.W,keys1,vals1));
    [tol,kasenums] = measure_tolerance_linear(bnd,pr,p0,vars,kasenums);
else
    [tol,kasenums] = measure_tolerance_nonlinear(bnd,pr,p0,vars,kasenums);
end

end


function [tol,kasenums] = measure_tolerance_linear(bnd,pr,p0,vars,kasenums)

keys = fieldnames(p0);
vals = struct2cell(p0);  % some param vals will be singular, some arrays
[xb,xi] = ismember(bnd.xi,keys);

yvals = log(cell2mat(vals(xi)));
tol = repmat(cell2struct(cell(size(vars)),vars,1),length(kasenums));
for i = 1:length(kasenums)
    for j = 1:length(vars)
        ki = kasenums(i);
        vb = strcmp(bnd.xi,vars{j});  % find the index of the desired var

        % Calculate the zeros (locations of the boundaries along the line).
        b = bnd.W(:,:,ki)*log(bnd.zeta(:,:,ki)) + log(bnd.delta(:,:,ki));
        A = bnd.U(:,~vb,ki);
        c = bnd.U(:,vb,ki);
        y = yvals(~vb);
        z = -(A*y + b)./c;
        
        % Add the additional parameter range values from pr.
        if isfield(pr,vars{j})
            r = pr.(vars{j});
            z = cat(1,z,log([min(r);max(r)]));
        end
        
        d = z - yvals(vb);
        tdown = 1/exp(max(d(d<0)));  % make it a fold change down
        tup = exp(min(d(d>0)));  % make it a fold change up

        if isempty(tdown)
            tdown = Inf;
        end
        if isempty(tup)
            tup = Inf;
        end
        
        tol(i).(vars{j}) = [tdown,tup];
    end
end

end


function varargout = measure_tolerance_nonlinear(varargin)

error(dspace.Settings.ERROR_NOT_IMPLEMENTED, ...
    'Function measure_tolerance_nonlinear not implemented.');

end




