function gma = parse_gma(f,xd)
% parse_gma  Parse equations in GMA form.
%
% Usage
%
%   gma = parse_gma(f,xd)
%
% Parameters
%
%   f: cell column vector of strings, equations in GMA form
%
%   xd: (optional) cell column vector of strings, dependent variables of the
%       system; default is {'X1';'X2';...;'X3'}
%
% Returns
%
%   gma: struct, GMA in standard form (see following fields)
%
%       gma.G,.H: any class 3-D array, kinetic orders of positive, negative
%                 terms (respectively) in a GMA system
%
%       gma.alpha,.beta: double matrix, numeric coefficients of positive,
%                        negative terms (respectively) in a GMA system;
%                        only the absolute value of the coefficient is used
%
%       gma.x: cell column vector of strings, dependent and independent
%              variables (including rate constants) of the system; first N
%              variables are the dependent variables
%
%       gma.xk: cell column vector of strings, variables in the kinetic orders
%
%       gma.sig: integer column vector, term signature in standard form
%                (see <a href="matlab:help dspace.gma.signature">signature</a>)

if nargin < 2
    m = length(f);
    xdpre = dspace.Settings.VARIABLE_PREFIX;
    xd = cellstr([repmat(xdpre,m,1),int2str((1:m)')]);
end

[k,sigma,gma.x,gma.xk] = dspace.gma.parse(f,xd);

% Force Expression to sym.
if isa(k,'dspace.Expression')
    k = sym(k);
end

[gma.G,gma.H,gma.alpha,gma.beta] = dspace.gma.split(k,sigma);
gma.sig = dspace.gma.signature(sigma);

end


