function h = plot_regions_2d(dat,reg,y1,y2)
% plot_regions_2d  Plot a set of sample points in 2-D.
%
% Usage
%
%   h = plot_regions_2d(dat,reg,y1,y2)
%
% Parameters
%
%   dat: 2-D double array, indexes into reg; N equals the number of pvals
%        with more than one value, ordered according to pvals field order;
%        dim 1 is oriented on the dependent axis, dim 2 is on the
%        independent axis
%
%   reg: boolean matrix, list of regions, or distinct combinations of valid
%        cases; columns represent cases, rows represent specific combinations
%        of valid cases
%
%   y1: double vector, values of the dependent axis in logspace
%
%   y2: double vector, values of the independent axis in logspace
%
% Returns
%
%   h: plot handle

h = image(y1,y2,dat.','CDataMapping','direct');
axis xy;

% Make [0,0,0,...] black in the colormap (if it exists).
z = zeros(1,size(reg,2));
[b,i] = ismember(z,reg,'rows');
if b
    map = hsv(size(reg,1) - 1);
    map = [map(1:i-1,:); [0,0,0]; map(i:end,:)];
    colormap(map);
else
    colormap(hsv(size(reg,1)));
end

% Add a colorbar.
key = unique(dat);
[regsort,i] = sortrows(~reg(key,:));
key = key(i);  % sort so colorbar labels start with region 1
i = (sum(~regsort,2) == 1);
key = [key(i);key(~i)];  % sort to start with non-intersecting regions
if b
    key = key(1:end-1);  % remove black from the colorbar
end

h1 = gca;
h2 = colorbar;
h1p = get(h1,'Position');
h2p = get(h2,'Position');
colorbar('off');
set(h1,'Position',h1p);
h3 = axes('Position',h2p);

image(key,'CDataMapping','direct');

% Create the label for the colorbar.
cases = (1:size(reg,2));
s = cell(size(key));
for i = 1:length(key)
    rn = cases(reg(key(i),:));
    si = num2str(rn,'%d,');
    if ~isempty(si)
        si = si(1:length(si)-1);  % remove trailing comma
    else
        si = 'none';
    end
    s{i} = si;
end
set(h3,'YTick',(1:size(reg,1)));
set(h3,'YTickLabel',s);
set(h3,'XTick',[]);
set(h3,'TickLength',[0,0]);
set(h3,'YAxisLocation','right');

axes(h1);  % return focus to original axes

end
