function [dat2,reg2] = remap_regions(dat,reg,reg0)
% remap_regions  Map sample points from one set of regions to another.
%
% Usage
%
%   [dat2,reg2] = remap_regions(dat,reg,reg0)
%
% Parameters
%
%   dat: N-D double array, indexes into reg; N equals the number of pvals
%        with more than one value, ordered according to pvals field order
%
%   reg: boolean matrix, list of regions, or distinct combinations of valid
%        cases; columns represent cases, rows represent specific combinations
%        of valid cases
%
%   reg0: (optional) boolean matrix, list of regions that already exist;
%                    default is reg (which should not change dat or reg)
%
% Returns
%
%   dat2: N-D double array, indexes into reg2
%
%   reg2: boolean matrix, list of regions that includes reg and reg0
%
% Notes
%
%   The function is useful when sampling and plotting serveral design
%   spaces.  A new image may not use the same set of regions as the old;
%   this function remaps the new to use the same set of regions as the old,
%   adds new regions as necessary (regions that only appear in the new
%   case), and returns an updated region list.

if nargin < 3
    reg0 = reg;
end

% Find all reg not in reg0 and append.  Remap dat to match new reg.
a = setdiff(reg, reg0, 'rows');
reg2 = [reg0;a];
dat2 = dat;
for i = 1:size(reg,1)
    [dummy,j] = intersect(reg2,reg(i,:),'rows');
    dat2(dat==i) = j;
end

end
