function fdat = sample_function(bnd,pvals,dat,reg,kases,f)
% sample_function  Sample arbitrary function using dominant S-systems.
%
% Usage
%
%   fdat = sample_function(kases,pvals,dat,reg,kases,f)
%
% Parameters
%
%   bnd: struct vector, design space boundaries in standard form
%        (see <a href="matlab:help dspace.calculate_boundaries">calculate_boundaries</a>)
%
%   pvals: struct, parameter values to be sampled; field names represent
%          parameter names and field values are vectors of values for each
%          parameter; all parameters must have at least one value
%
%   dat: N-D double array, indexes into reg; N equals the number of pvals
%        with more than one value, ordered according to pvals field order
%
%   reg: boolean matrix, list of regions, or distinct combinations of valid
%        cases; columns represent cases, rows represent specific combinations
%        of valid cases
%
%   kases: struct vector, design space cases in standard form
%          (see <a href="matlab:help dspace.enumerate_cases">enumerate_cases</a>)
%
%   f: function handle, function that operates on dominant S-system in the
%      form f(G0,H0,alpha0,beta0,xi0,xk0,dG0,dH0)
%
% Returns
%
%   fdat: cell array of size dat, each element is a cell array of function
%         values for each dominant S-system of the valid regions at the point

% Test to see if the kinetic orders are pinned to a single value.
keys = fieldnames(pvals);
vals = struct2cell(pvals);
i1 = (cellfun(@length,vals) == 1);
keys1 = keys(i1);
vals1 = vals(i1);

if isempty(bnd.xk)
    fdat = sample_function_linear(bnd,pvals,dat,reg,kases,f);    
elseif all(ismember(bnd.xk,keys1))
    bnd.U = double(dspace.util.safe_subs(bnd.U,keys1,vals1));
    bnd.W = double(dspace.util.safe_subs(bnd.W,keys1,vals1));
    fdat = sample_function_linear(bnd,pvals,dat,reg,kases,f);
else
    fdat = sample_function_nonlinear(bnd,pvals,dat,reg,kases,f);
end

end


function fdat = sample_function_linear(bnd,pvals,dat,reg,kases,f)

% Get xk0, dG0, dH0, which are constant over the design space.
xk0 = cell2mat(dspace.util.getfields(pvals,bnd.xk));
G0all = kases.G;
H0all = kases.H;
dG0all = zeros([size(kases.G),length(bnd.xk)]);
dH0all = dG0all;
if ~isempty(bnd.xk)
    G0all = double(dspace.util.safe_subs(kases.G,bnd.xk,xk0));
    H0all = double(dspace.util.safe_subs(kases.H,bnd.xk,xk0));
    for i = 1:length(bnd.xk)
        dG0all(:,:,:,i) = double(dspace.util.safe_subs(diff(kases.G,bnd.xk{i}),bnd.xk,xk0));
        dH0all(:,:,:,i) = double(dspace.util.safe_subs(diff(kases.H,bnd.xk{i}),bnd.xk,xk0));
    end
end
dG0all = permute(dG0all,[1,2,4,3]);  % put case num in last position
dH0all = permute(dH0all,[1,2,4,3]);  % put case num in last position
    
% Only keep pvals fields that are in xi.
pkeys = fieldnames(pvals);
pvals = rmfield(pvals,pkeys(~ismember(pkeys,bnd.xi)));
p0 = struct2cell(pvals);

% Get all combinations of pvals.
pp = dspace.util.vector_combinations(p0{:});
keys = fieldnames(pvals);
[b,keysi] = ismember(bnd.xi,keys);  % reverse mapping from pvals to xi
xx = pp(keysi,:);

% Eval at every point.
fdat = cell(size(dat));
for i = 1:numel(dat)
    xi0 = xx(:,i);  % get pvals for the point
    regi = find(reg(dat(i),:));  % get valid regions for point
    fdat{i} = cell(length(regi),1);  % init results for point
    for j = 1:length(regi)
        alpha0 = kases.alpha(:,:,regi(j));
        beta0 = kases.beta(:,:,regi(j));
        G0 = G0all(:,:,regi(j));
        H0 = H0all(:,:,regi(j));
        dG0 = dG0all(:,:,:,regi(j));
        dH0 = dH0all(:,:,:,regi(j));
        fdat{i}{j} = f(G0,H0,alpha0,beta0,xi0,xk0,dG0,dH0);  % eval at point
    end
end

end


function varargout = sample_function_nonlinear(varargin)

error(dspace.Settings.ERROR_NOT_IMPLEMENTED, ...
    'Function sample_function_nonlinear not implemented.');

end


