function [dat,reg] = sample_regions(bnd,pvals)
% sample_regions  Sample point by point for valid cases.
%
% Usage
%
%   [dat,reg] = sample_regions(bnd,pvals)
%
% Parameters
%
%   bnd: struct vector, design space boundaries in standard form
%        (see <a href="matlab:help dspace.calculate_boundaries">calculate_boundaries</a>)
%
%   pvals: struct, parameter values to be sampled; field names represent
%          parameter names and field values are vectors of values for each
%          parameter; all parameters must have at least one value
%
% Returns
%
%   dat: N-D double array, indexes into reg; N equals the number of pvals
%        with more than one value, ordered according to pvals field order
%
%   reg: boolean matrix, list of regions, or distinct combinations of valid
%        cases; columns represent cases, rows represent specific combinations
%        of valid cases

% Test to see if the kinetic orders are pinned to a single value.
keys = fieldnames(pvals);
vals = struct2cell(pvals);
i1 = (cellfun(@length,vals) == 1);
keys1 = keys(i1);
vals1 = vals(i1);

if isempty(bnd.xk)
    [dat,reg] = sample_regions_linear(bnd,pvals);    
elseif all(ismember(bnd.xk,keys1))
    bnd.U = double(dspace.util.safe_subs(bnd.U,keys1,vals1));
    bnd.W = double(dspace.util.safe_subs(bnd.W,keys1,vals1));
    [dat,reg] = sample_regions_linear(bnd,pvals);
else
    [dat,reg] = sample_regions_nonlinear(bnd,pvals);
end

end


function [dat,reg] = sample_regions_linear(bnd,pvals)
% Create a set of points and test each to see if the parameters meet the
% constraints for each case.
% m = num constraints; n = number of variables/parameters; p = num cases
% NOTE: pr should have two variables with ranges, the rest single values.
% NOTE: should speed things up by determining validity and only testing
% valid cases (and then adding zeros in afterwards).

% Keep and reorder elements of pr to match x.
keys = fieldnames(pvals);
vals = struct2cell(pvals);
[xb,xi] = ismember(bnd.xi,keys);
if ~all(xb)
    error(dspace.Settings.ERROR_MISSING_PARAMETERS, ...
        'Missing parameters.');
end
xvals = vals(xi);
xlens = cellfun(@length,xvals);
if any(xlens == 0)
    error(dspace.Settings.ERROR_MISSING_PARAMETERS, ...
        'Some parameter values are empty.');
end

% Extract fixed values.
x0b = (xlens == 1);  % indices of params with one value
x0 = cat(1,xvals{x0b});

% Create every combination of values from params with multiple values.
n = xlens(~x0b);  % number of samples in each variable
xx = dspace.util.vector_combinations(xvals{~x0b});

% Put into logspace.
y0 = log(x0);
yy = log(xx);
z = log(bnd.zeta);
d = log(bnd.delta);

% Loop through all cases.
p = size(bnd.U,3);  % number of a cases
t = false(prod(n),p);  % initialize test results
for j = 1:p
    b = bnd.W(:,:,j)*z(:,:,j) + d(:,:,j);
    if ~isempty(y0)
        b = b + bnd.U(:,x0b,j)*y0;
    end
    t(:,j) = all(bnd.U(:,~x0b,j)*yy + repmat(b,1,prod(n)) > 0);
end

reg = unique(t,'rows');  % unique combinations of cases
dat = zeros(size(t,1),1);
for i = 1:size(reg,1)
    j = all(t == repmat(reg(i,:),size(t,1),1),2);
    dat(j,:) = i;
end
    
dat = reshape(dat,n');

% Permute dimensions of the results to match order of params in pr.
[kb,ki] = ismember(fieldnames(pvals),keys(xi(~x0b)));
dat = dspace.util.safe_permute(dat,ki(kb));

end


function varargout = sample_regions_nonlinear(varargin)

error(dspace.Settings.ERROR_NOT_IMPLEMENTED, ...
    'Function sample_regions_nonlinear not implemented.');

end
