%% Design Space Example 1
% This simple example demonstrates the basic steps and functions necessary
% to construct and analyze a design space using the Desigh Space Toolbox.

%% Enter Equations
% The following system has two dependent variables and two independent
% variables.  Ultimately, the design space will vary over the two
% independent variables X3 and X4.
%
% dX1/dt =     10*X1*X2*X3*X4 + X1*X2 - X1
% dX2/dt = (1/10)*X1*X2*X3*X4 + X1*X2 - X2
%
% The equations are entered as a cell array of strings.

f = {'    10*X1*X2*X3*X4 + X1*X2 - X1';
     '(1/10)*X1*X2*X3*X4 + X1*X2 - X2'};

%% Enumerate Cases
% A design space is completely characterized by a set of dominant S-systems
% and the conditions for their dominance.  Boundaries are the conditions in
% terms of the independent variables.
% 
% The following steps are common in the analysis of design space: parse the
% equations, enumerate the cases, and calculate the boundaries.

gma = dspace.parse_gma(f);
cas = dspace.enumerate_cases(gma);
bnd = dspace.calculate_boundaries(cas);

%%
% The term signature, or number of positive and negative terms in each
% equation, is important throughout the analysis.

disp(f);
disp(gma.sig');

%%
% The dominant S-systems and conditions for each case can be printed to the
% command window.  Here, only the first case is shown.

dspace.print.print_cases_pretty(cas,1);

%% Test for Validity
% Some sets of boundaries are mutually exclusive, or invalid.  Under certain
% conditions, the validity of boundaries can be checked automatically.

v = dspace.is_valid(bnd);
disp(find(v)');  % valid case numbers

%%
% The dominant terms in each case can be printed to the command window.
% Here, only the valid cases are shown.

dspace.print.print_dominant_terms(gma.sig,find(v));

%% Plot
% Regions in design space are made up of one or more overlapping cases.  To
% visualize the regions, individual points are checked to see if they lie
% inside the boundaries of each case.
%
% First, the points to test must be designated.  Here, X3 and X4 are each
% assigned 500 evenly spaced points (in in logarithmic coordinates),
% implying a 500x500 grid of parameter combinations to test.

pvals = struct();  % param values
pvals.X3 = 10.^linspace(-3,3,500);
pvals.X4 = 10.^linspace(-3,3,500);

%%
% Then, the points are tested and plotted, showing the three unique
% regions in design space.  The two axes are X3 and X4, the independent
% variables in the original equations.

clf;
[dat,reg] = dspace.sample_regions(bnd,pvals);
dspace.plot_regions_2d(dat,reg,log10(pvals.X3),log10(pvals.X4));
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
grid on;
