%% Design Space Example 2
% This example is similar to the previous example, but demonstrates more
% advanced functions of the Design Space Toolbox.

%% Enter Equations
% The following system is similar to the previous example, but with
% multiple negative terms.  It still has two dependent variables and two
% independent variables, and the design space will vary over the two
% independent variables X3 and X4.
%
% dX1/dt =     10*X1*X2*X3*X4 + X1*X2 -     10*X1*X3*X4^-1 - X1
% dX1/dt = (1/10)*X1*X2*X3*X4 + X1*X2 - (1/10)*X2*X3*X4^-1 - X2
%
% The equations are entered as a cell array of strings.  Here, the
% dependent variables are specified explicitly.

f = {'    10*X1*X2*X3*X4 + X1*X2 -     10*X1*X3*X4^-1 - X1';
     '(1/10)*X1*X2*X3*X4 + X1*X2 - (1/10)*X2*X3*X4^-1 - X2'};
xd = {'X1';'X2'};

%% Enumerate Cases
% The following steps are common in the analysis of design space: parse the
% equations, enumerate the cases, and calculate the boundaries.

gma = dspace.parse_gma(f,xd);
cas = dspace.enumerate_cases(gma);
bnd = dspace.calculate_boundaries(cas);

%% Test Validity
% Cases are checked to see if they are valid and the dominant terms are
% printed for each valid case.

v = dspace.is_valid(bnd);
vn = find(v);  % valid case numbers
dspace.print.print_dominant_terms(bnd.sig,vn);

%% Plot
% Points are tested to see if they lie inside the boundaries for each case
% and the results are plotted.  Here, X3 and X4 are each assigned 500
% evenly spaced points (in logarithmic coordinates), implying a 500x500
% grid of combinations to test.

pvals = struct();  % param vals
pvals.X3 = 10.^linspace(-3,3,500);
pvals.X4 = 10.^linspace(-3,3,500);

figure(1);
clf;

[dat,reg] = dspace.sample_regions(bnd,pvals);
dspace.plot_regions_2d(dat,reg,log10(pvals.X3),log10(pvals.X4));
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
grid on;

%% Create a Bounding Box
% A bounding box is calculated for case 4, the diamond in the center.  The
% result is given as lower and upper bounds, respectively, on the
% parameters, X3 and X4 (in Cartesian coordinates).

box = dspace.create_boundingbox(bnd);
disp(box(4));

%%
% The box is plotted in the previous figure, where the bounds (in
% logarithmic coordinates) can be visually confirmed.

hold on;
y3b = log10(box(4).X3);
y4b = log10(box(4).X4);
y3p = [y3b(1),y3b(2),y3b(2),y3b(1)];
y4p = [y4b(1),y4b(1),y4b(2),y4b(2)];
patch(y3p,y4p,'k','FaceColor','none');  % plot box
hold off;

disp('log10(X) = ')
disp(structfun(@log10,box(4),'UniformOutput',false));

%% Measure Tolerance
% An operating point is set in case 4 and tolerance is measured to the
% boundaries.  The result is given as fold change down and fold change up,
% respecively (in Cartesian coordinates).

p0 = struct();  % operating point
p0.X3 = 2;
p0.X4 = 1;
tol = dspace.measure_tolerance(bnd,p0);
disp(tol);

%%
% The operating point is plotted in the previous figure, where the
% tolerance (in logarithmic coordinates) can be visually confirmed.

hold on;
y30 = log10(p0.X3);
y40 = log10(p0.X4);
plot(y30,y40,'k.');  % plot operating point
line([y30,y30+log10(tol.X3(1))],[y40,y40],'Color','k');  % plot tol
line([y30,y30+log10(tol.X3(2))],[y40,y40],'Color','k');  % plot tol
line([y30,y30],[y40,y40+log10(tol.X4(1))],'Color','k');  % plot tol
line([y30,y30],[y40,y40+log10(tol.X4(2))],'Color','k');  % plot tol
hold off;

disp('log10(X) = ')
disp(structfun(@log10,tol,'UniformOutput',false));

%% Change the System
% The width of the regions are determined by the value 10 found in the rate
% constants of the original equations.  That value can be varied by adding
% a new parameter, alpha, and repeating the analysis.  Note that most of
% the cases in the new system are valid.

f = {'    alpha*X1*X2*X3*X4 + X1*X2 -     alpha*X1*X3*X4^-1 - X1';
     '(1/alpha)*X1*X2*X3*X4 + X1*X2 - (1/alpha)*X2*X3*X4^-1 - X2'};
xd = {'X1';'X2'};

gma = dspace.parse_gma(f,xd);
cas = dspace.enumerate_cases(gma);
bnd = dspace.calculate_boundaries(cas);

v = dspace.is_valid(bnd);
vn = find(v);  % valid case numbers
dspace.print.print_dominant_terms(bnd.sig,vn);

%% Add Contraints to Parameters
% Additional constraints can be placed on the parameters of the system.
% Here, validity is tested again while forcing alpha to the original value
% of 10. Note that the valid cases are now the same as those in the
% original system.

pr = struct();  % param ranges
pr.alpha = 10;

v = dspace.is_valid(bnd,pr);
vn = find(v);
dspace.print.print_dominant_terms(bnd.sig,vn);

%% Plot Multiple Cases
% The design space is plotted for two cases: alpha = 10 and alpha = 1/10.

figure(2);
clf;

subplot(121);
pvals.alpha = 10;
[dat1,reg1] = dspace.sample_regions(bnd,pvals);
dspace.plot_regions_2d(dat1,reg1,log10(pvals.X3),log10(pvals.X4));
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
title('\alpha = 10');
grid on;

subplot(122);
pvals.alpha = 1/10;
[dat2,reg2] = dspace.sample_regions(bnd,pvals);
dspace.plot_regions_2d(dat2,reg2,log10(pvals.X3),log10(pvals.X4));
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
title('\alpha = 1/10');
grid on;

%%
% Note that the previous plot coloring is inconsistent: in certain
% instances the plots use the same color for different cases. Each plot is
% sampled and drawn separately, coloring the unique regions as they are
% encoutered.  To coordinate the colors, the list of regions must be
% combined and the data updated to reflect the combined list.

[dat2b,reg2b] = dspace.remap_regions(dat2,reg2,reg1);  % remap the 2nd plot

figure(3);
clf;

subplot(121);
dspace.plot_regions_2d(dat1,reg1,log10(pvals.X3),log10(pvals.X4));
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
title('\alpha = 10');
grid on;

subplot(122);
dspace.plot_regions_2d(dat2b,reg2b,log10(pvals.X3),log10(pvals.X4)); % remapped
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
title('\alpha = 1/10');
grid on;

%% Plot Slices 
% A series is plotted with alpha changing from 10 to 1/10.  Note that
% combinations of alpha, X3, and X4 produce a 3-D, or 200x200x20, array of
% results.  The loop extracts and plots 2-D slices of the 3-D array.

figure(4);
clf;

pvals = struct();
pvals.X3 = 10.^linspace(-3,3,200);
pvals.X4 = 10.^linspace(-3,3,200);
pvals.alpha = 10.^fliplr(linspace(-1,1,20));  % multiple samples for alpha

[dat,reg] = dspace.sample_regions(bnd,pvals);
[dat,reg] = dspace.remap_regions(dat,reg,reg2b);  % remap to figure(3)

for i = 1:length(pvals.alpha)
    clf;
    dspace.plot_regions_2d(dat(:,:,i),reg,log10(pvals.X3),log10(pvals.X4));
    xlabel('log_{10}(X_3)');
    ylabel('log_{10}(X_4)');
    title(['\alpha = ',num2str(log10(pvals.alpha(i)),3)]);
    grid on;
    drawnow;
    snapnow;
    pause(0.1);
end
