%% Design Space Example 3
% This example demonstrates how the Design Space Toolbox handles
% intersecting, or overlapping, cases, which can indicate the potential
% for hysteresis.

%% Enter Equations
% The following system has two dependent variables and two independent
% variables.  Ultimately, the design space will vary over the two
% independent variables X3 and X4.
%
%   dX1/dt = 1000*X1^2*X2^-1 + 1000*X2^-1 - X1*X3*X4
%   dX2/dt = X1^2 + 10000 - X2
%
% The equations and dependent variables are entered as strings.

f = {'1000*X1^2*X2^-1 + 1000*X2^-1 - X1*X3*X4';
     'X1^2 + 10000 - X2'};
xd = {'X1';'X2'};

%% Enumerate Cases
% The following steps are common in the analysis of design space: parse the
% equations, enumerate the cases, and calculate the boundaries.

gma = dspace.parse_gma(f,xd);
cas = dspace.enumerate_cases(gma);
bnd = dspace.calculate_boundaries(cas);

%% Test Validity
% Cases are checked to see if they are valid and the dominant terms are
% printed for each valid case.

v = dspace.is_valid(bnd);
vn = find(v);  % valid case numbers
dspace.print.print_dominant_terms(bnd.sig,vn);

%% Search for Intersections
% Under certain conditions, the design space can be exhaustively searched
% for all occurances of intersecting cases.  Occurances of three
% intersecting cases are especially interesting--they indicate the
% potential for hysteresis.  Here, the one occurance of three intersecting
% cases (1, 2, and 4) is displayed.

int = dspace.find_intersections(bnd);
disp(int{3});

%% Plot
% Points are tested to see if they lie inside the boundaries for each case
% and the results are plotted.  Here, X3 and X4 are each assigned 500
% evenly spaced points (in in logarithmic coordinates), implying a 500x500
% grid of combinations to test.  Note that one region represents the
% intersection of three cases (1, 2, and 4).

pvals = struct();  % param vals
pvals.X3 = 10.^linspace(-3,3,500);
pvals.X4 = 10.^linspace(-3,3,500);

figure(1);
clf;

[dat,reg] = dspace.sample_regions(bnd,pvals);
dspace.plot_regions_2d(dat,reg,log10(pvals.X3),log10(pvals.X4));
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
grid on;
