%% Design Space Example 4
% This example demonstrates how the Design Space Toolbox handles cases that
% are invalid due to rank deficiency and regions of design space that do not
% satisfy the conditions of any case.

%% Enter Equations
% The following system has two dependent variables and two independent
% variables.  Ultimately, the design space will vary over the two
% independent variables alpha and beta.
%
%   dX1/dt = X1*X2 + X2 - alpha*X1
%   dX2/dt = X1*X2 + X1 - beta*X2
%
% The equations and dependent variables are entered as strings.

f = {'X1*X2 + X2 - alpha*X1';
     'X1*X2 + X1 - beta*X2'};
xd = {'X1';'X2'};

%% Enumerate Cases
% The following steps are common in the analysis of design space: parse the
% equations, enumerate the cases, and calculate the boundaries.

gma = dspace.parse_gma(f,xd);
cas = dspace.enumerate_cases(gma);
bnd = dspace.calculate_boundaries(cas);

%% Test Validity
% Cases are checked to see if they are valid and the dominant terms are
% printed for each valid case.  Note that in case 4, when the second
% positive term dominates in both equations, the system is rank deficient
% and declared invalid.

v = dspace.is_valid(bnd);
vn = find(v);  % valid case numbers
dspace.print.print_dominant_terms(bnd.sig,vn);


%% Plot
% Points are tested to see if they lie inside the boundaries for each case
% and the results are plotted.  Here, alpha and beta are each assigned 500
% evenly spaced points (in in logarithmic coordinates), implying a 500x500
% grid of combinations to test.  Note that some points do not lie inside
% the boundaries of any case, and are therefore black.

pvals = struct();  % param vals
pvals.alpha = 10.^linspace(-3,3,500);
pvals.beta = 10.^linspace(-3,3,500);

figure(1);
clf;

[dat,reg] = dspace.sample_regions(bnd,pvals);
dspace.plot_regions_2d(dat,reg,log10(pvals.alpha),log10(pvals.beta));
xlabel('log_{10}(\alpha)');
ylabel('log_{10}(\beta)');
grid on;

%% Create a Bounding Box
% A bounding box is calculated for case 1 while limiting the parameters to
% the range of the plot.  The result is given as lower and upper bounds,
% respectively, on the parameters, alpha and beta (in Cartesian coordinates).

box = dspace.create_boundingbox(bnd,pvals);
disp(box(1));

%%
% If the parameters are not limited to the range of the plot, there are no
% upper bounds on case 1.

box = dspace.create_boundingbox(bnd);
disp(box(1));

%% Measure Tolerance
% An operating point is set in case 1 and tolerance is measured to the
% boundaries. The result is given as fold change down and fold change up,
% respecively (in Cartesian coordinates).

p0 = struct();  % operating point
p0.alpha = 10;
p0.beta = 10;
tol = dspace.measure_tolerance(bnd,p0);
disp(tol);
