%% Design Space Example 5
% This simple example demonstrates the analysis of the dominant S-system
% in each region using the Design Space Toolbox.

%% Enter Equations
% The following system was introduced in Design Space Example 1.  It has
% two dependent variables and two independent variables.  Ultimately, the
% design space will vary over the two independent variables X3 and X4.
%
% dX1/dt =     10*X1*X2*X3*X4 + X1*X2 - X1
% dX1/dt = (1/10)*X1*X2*X3*X4 + X1*X2 - X2
%
% Here, the system is input, parsed, and analyzed.

f = {'    10*X1*X2*X3*X4 + X1*X2 - X1';
     '(1/10)*X1*X2*X3*X4 + X1*X2 - X2'};

gma = dspace.parse_gma(f);
cas = dspace.enumerate_cases(gma);
bnd = dspace.calculate_boundaries(cas);

%% Plot Regions
% Here, the regions are plotted as in the other examples.  X3 and X4 are
% each assigned 100 evenly spaced points (in in logarithmic coordinates),
% implying a 100x100 grid of parameter combinations to test.

pvals = struct();  % param values
pvals.X3 = 10.^linspace(-3,3,100);
pvals.X4 = 10.^linspace(-3,3,100);

figure(1);
clf;

[dat,reg] = dspace.sample_regions(bnd,pvals);
dspace.plot_regions_2d(dat,reg,log10(pvals.X3),log10(pvals.X4));
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
grid on;

%% Sample the Steady States
% In every region, the dominant S-system can be analyzed using well-known
% methods from Biochemical Systems Theory and the Power Law Formalism.
% Here, the steady state is calculated at each point.

X10 = dspace.ssys.sample_steady_state(bnd,pvals,dat,reg,cas,'X1');
X20 = dspace.ssys.sample_steady_state(bnd,pvals,dat,reg,cas,'X2');

%% Plot the Steady States
% The data is returned as a cell array.  Each element represents a point in
% design space.  Each element is a cell array that contains zero or more
% entries, one for each valid region at that point.  Extracting the data
% requires knowledge of the regions and appropriate MATLAB commands.  Here,
% the steady state for the first, and only, dominant region is extracted and
% split between values for X1 and X2.  The resulting double matrices are
% plotted.

X10 = cellfun(@(x)x{:},X10);  % get the only value
X20 = cellfun(@(x)x{:},X20);  % get the only value

figure(2);
clf;

subplot(121);
dspace.plot_function_2d(X10,log10(pvals.X3),log10(pvals.X4));
title('X_1 (approximate)');
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
grid on;

subplot(122);
dspace.plot_function_2d(X20,log10(pvals.X3),log10(pvals.X4));
title('X_2 (approximate)');
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
grid on;

%% Determine the Exact Values
% Here, the steady states are solved for analytically and plotted for
% comparision.

sol = solve(sym(f));
fX1 = matlabFunction(sol.X1(sol.X1~=0));
fX2 = matlabFunction(sol.X2(sol.X2~=0));
[X3,X4] = ndgrid(pvals.X3,pvals.X4);
X1 = fX1(X3,X4);
X2 = fX2(X3,X4);

figure(3);
clf;

subplot(121);
dspace.plot_function_2d(X1,log10(pvals.X3),log10(pvals.X4));
title('X_1 (actual)');
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
grid on;

subplot(122);
dspace.plot_function_2d(X2,log10(pvals.X3),log10(pvals.X4));
title('X_2 (actual)');
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
grid on;

%% Compare the Results
% Here, the relative error between the approximate and exact values is
% calculated.  Note that the greatest error occurs at the boundaries
% between regions.

errX1 = abs(X1-X10)./X1;
errX2 = abs(X2-X20)./X2;

figure(4);
clf;

subplot(121);
dspace.plot_function_2d(errX1,log10(pvals.X3),log10(pvals.X4));
title('X_1 (relative error)');
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
grid on;

subplot(122);
dspace.plot_function_2d(errX2,log10(pvals.X3),log10(pvals.X4));
title('X_2 (relative error)');
xlabel('log_{10}(X_3)');
ylabel('log_{10}(X_4)');
grid on;
