function test_suite = test_dspace_boundaries_1
    initTestSuite;
end

function dat = setup
    % Create simple boundaries.
    dat.bnd.U = cat(3,[1,1;1,1],[1,1;1,1]);
    dat.bnd.delta = exp(cat(3,[1;1],[1;-1]));
    
    dat.bnd.W = zeros(2,2,2);
    dat.bnd.zeta = exp(zeros(2,1,2));

    dat.bnd.xi = sym({'x3';'x4'});  % independent variables
    dat.bnd.xk = {};
    
    % Vary the kinetic orders, but pin them.
    dat.bndsym = dat.bnd;
    dat.bndsym.U = sym(cat(3,{'a','b';'c','d'},{'e','f';'g','h'}));
    dat.bndsym.W = sym(zeros(2,2,2));
    dat.bndsym.xk = {'a';'b';'c';'d';'e';'f';'g';'h'};
    
    dat.p0.a = 1;
    dat.p0.b = 1;
    dat.p0.c = 1;
    dat.p0.d = 1;
    dat.p0.e = 1;
    dat.p0.f = 1;
    dat.p0.g = 1;
    dat.p0.h = 1;
end

% =====
% Test validity.

function test_is_valid_1(dat)
    f = dspace.is_valid(dat.bnd);
    assertEqual(f,[true;true]);
end

function test_is_valid_2(dat)
% Create opposing constraints (i.e. A*y > b, A*y < b) in case 2.  
    dat.bnd.U(2,:,2) = -dat.bnd.U(2,:,2);
    f = dspace.is_valid(dat.bnd);
    assertEqual(f,[true;false]);
end

% Repeat for symbolic case.

function test_is_valid_1_sym(dat)
    f = dspace.is_valid(dat.bndsym,dat.p0);
    assertEqual(f,[true;true]);
end

function test_is_valid_2_sym(dat)
% Create opposing constraints (i.e. A*y > b, A*y < b) in case 2.  
    dat.bndsym.U(2,:,2) = -dat.bndsym.U(2,:,2);
    f = dspace.is_valid(dat.bndsym,dat.p0);
    assertEqual(f,[true;false]);
end

% Show not implemented for incomplete symbolic case.

function test_is_valid_err_sym(dat)
% Leave a variable in kinetic orders (nonlinear case).
    p0 = rmfield(dat.p0,'h');
    f = @()dspace.is_valid(dat.bndsym,p0);
    assertExceptionThrown(f,dspace.Settings.ERROR_NOT_IMPLEMENTED);
end

% =====
% Test intersection.

function test_is_intersecting_1(dat)
    f = dspace.is_intersecting(dat.bnd,[],[1;2]);
    assertEqual(f,true);
end

function test_is_intersecting_2(dat)
% Create opposing constraints (i.e. A*y > b, A*y < b) in case 2.  
    dat.bnd.U(2,:,2) = -dat.bnd.U(2,:,2);
    f = dspace.is_intersecting(dat.bnd,[],[1;2]);
    assertEqual(f,false);
end

% Repeat for the symbolic case.

function test_is_intersecting_1_sym(dat)
    f = dspace.is_intersecting(dat.bndsym,dat.p0,[1;2]);
    assertEqual(f,true);
end

function test_is_intersecting_2_sym(dat)
% Create opposing constraints (i.e. A*y > b, A*y < b) in case 2.  
    dat.bndsym.U(2,:,2) = -dat.bndsym.U(2,:,2);
    f = dspace.is_intersecting(dat.bndsym,dat.p0,[1;2]);
    assertEqual(f,false);
end

% Show not implemented for incomplete symbolic case.

function test_is_intersecting_err_sym(dat)
% Leave a variable in kinetic orders (nonlinear case).
    p0 = rmfield(dat.p0,'h');
    f = @()dspace.is_intersecting(dat.bndsym,p0,[1;2]);
    assertExceptionThrown(f,dspace.Settings.ERROR_NOT_IMPLEMENTED);
end



