function test_suite = test_dspace_boundaries_2
    initTestSuite;
end

function dat = setup
    % Sample system.
    dat.bnd.U = cat(3,[ 1, 1;-1,-1], ...
                      [-1, 1; 1,-1], ...
                      [ 0, 1; 0,-1], ...
                      [ 1, 0;-1, 0], ...
                      [ 1, 0;-1, 0]);
    dat.bnd.delta = exp(cat(3,[1;1],[1;1],[1;1],[1;1],[-3;5]));

    dat.bnd.W = zeros(2,2,5);
    dat.bnd.zeta = exp(zeros(2,1,5));
    
    dat.bnd.xi = {'x3';'x4'};  % independent variables
    dat.bnd.xk = {};
    
    % Symbolic version.
    dat.bndsym = dat.bnd;
    dat.bndsym.U = sym(cat(3,{'a1','b1';'c1','d1'}, ...
                           {'a2','b2';'c2','d2'}, ...
                           {'a3','b3';'c3','d3'}, ...
                           {'a4','b4';'c4','d4'}, ...
                           {'a4','b4';'c4','d4'}));
    dat.bndsym.W = sym(zeros(2,2,5));
    dat.bndsym.xk = {'a1';'a2';'a3';'a4';
                   'b1';'b2';'b3';'b4';
                   'c1';'c2';'c3';'c4';
                   'd1';'d2';'d3';'d4'};
    
    dat.p0.a1 = 1;
    dat.p0.b1 = 1;
    dat.p0.c1 = -1;
    dat.p0.d1 = -1;

    dat.p0.a2 = -1;
    dat.p0.b2 = 1;
    dat.p0.c2 = 1;
    dat.p0.d2 = -1;
    
    dat.p0.a3 = 0;
    dat.p0.b3 = 1;
    dat.p0.c3 = 0;
    dat.p0.d3 = -1;
    
    dat.p0.a4 = 1;
    dat.p0.b4 = 0;
    dat.p0.c4 = -1;
    dat.p0.d4 = 0;
    
    % Known intersections.
    dat.kk = cell(4,1);
    dat.kk{1} = [1;2;3;4;5];
    dat.kk{2} = [1,2;1,3;1,4;1,5;2,3;2,4;2,5;3,4;3,5];
    dat.kk{3} = [1,2,3;1,2,4;1,3,4;2,3,4];
    dat.kk{4} = [1,2,3,4];
end

% =====
% Test is_intersecting.

function test_is_intersecting_1(dat)
    f = dspace.is_intersecting(dat.bnd,[],(1:5)');
    assertEqual(f,false);
end

function test_is_intersecting_2(dat)
    f = dspace.is_intersecting(dat.bnd,[],(1:4)');
    assertEqual(f,true);
end

function test_is_intersecting_3(dat)
    f = dspace.is_intersecting(dat.bnd,[],[4;5]);
    assertEqual(f,false);
end

function test_is_intersecting_4(dat)
    f = dspace.is_intersecting(dat.bnd,[],[3;5]);
    assertEqual(f,true);
end

% Repeat for symbolic case.

function test_is_intersecting_1_sym(dat)
    f = dspace.is_intersecting(dat.bndsym,dat.p0,(1:5)');
    assertEqual(f,false);
end

function test_is_intersecting_2_sym(dat)
    f = dspace.is_intersecting(dat.bndsym,dat.p0,(1:4)');
    assertEqual(f,true);
end

function test_is_intersecting_3_sym(dat)
    f = dspace.is_intersecting(dat.bndsym,dat.p0,[4;5]);
    assertEqual(f,false);
end

function test_is_intersecting_4_sym(dat)
    f = dspace.is_intersecting(dat.bndsym,dat.p0,[3;5]);
    assertEqual(f,true);
end

% Show not implemented for incomplete symbolic case.

function test_is_intersecting_err_sym(dat)
% Leave a variable in kinetic orders (nonlinear case).
    p0 = rmfield(dat.p0,'a1');
    f = @()dspace.is_intersecting(dat.bndsym,p0,[3;5]);
    assertExceptionThrown(f,dspace.Settings.ERROR_NOT_IMPLEMENTED);
end

% =====
% Test find_intersections.

function test_find_intersections_1(dat)
    kk = dspace.find_intersections(dat.bnd,[],1);
    assertEqual(kk,dat.kk(1));
end

function test_find_intersections_2(dat)
    kk = dspace.find_intersections(dat.bnd,[],2); 
    assertEqual(kk,dat.kk(2));
end

function test_find_intersections_3(dat)
    kk = dspace.find_intersections(dat.bnd,[],3); 
    assertEqual(kk,dat.kk(3));
end

function test_find_intersections_4(dat)
    kk = dspace.find_intersections(dat.bnd,[],4); 
    assertEqual(kk,dat.kk(4));
end

function test_find_intersections_5(dat)
    kk = dspace.find_intersections(dat.bnd,[],5); 
    assertEqual(kk,{});
end

function test_find_intersections_all(dat)
    kk = dspace.find_intersections(dat.bnd,[]);
    assertEqual(kk,dat.kk);
end

% Repeat for symbolic case.

function test_find_intersections_1_sym(dat)
    kk = dspace.find_intersections(dat.bndsym,dat.p0,1);
    assertEqual(kk,dat.kk(1));
end

function test_find_intersections_2_sym(dat)
    kk = dspace.find_intersections(dat.bndsym,dat.p0,2); 
    assertEqual(kk,dat.kk(2));
end

function test_find_intersections_3_sym(dat)
    kk = dspace.find_intersections(dat.bndsym,dat.p0,3); 
    assertEqual(kk,dat.kk(3));
end

function test_find_intersections_4_sym(dat)
    kk = dspace.find_intersections(dat.bndsym,dat.p0,4); 
    assertEqual(kk,dat.kk(4));
end

function test_find_intersections_5_sym(dat)
    kk = dspace.find_intersections(dat.bndsym,dat.p0,5); 
    assertEqual(kk,{});
end

function test_find_intersections_all_sym(dat)
    kk = dspace.find_intersections(dat.bndsym,dat.p0);
    assertEqual(kk,dat.kk);
end

% Show not implemented for incomplete symbolic case.

function test_find_intersections_err_sym(dat)
% Leave a variable in kinetic orders (nonlinear case).
    p0 = rmfield(dat.p0,'a1');
    f = @()dspace.find_intersections(dat.bndsym,p0);
    assertExceptionThrown(f,dspace.Settings.ERROR_NOT_IMPLEMENTED);
end

% =====
% Test create_boundingbox.

function test_create_boundingbox_1(dat)
% Combine test regions 3 and 4.
    bnd = dspace.combine_boundaries(dat.bnd,[3;4]);
    box = dspace.create_boundingbox(bnd);
    assertElementsAlmostEqual(box.x3,exp([-1,1]));
    assertElementsAlmostEqual(box.x4,exp([-1,1]));
end

function test_create_boundingbox_2(dat)
% Combine test regions 1 and 2.
    bnd = dspace.combine_boundaries(dat.bnd,[1;2]);
    box = dspace.create_boundingbox(bnd);
    assertElementsAlmostEqual(box.x3,exp([-1,1]));
    assertElementsAlmostEqual(box.x4,exp([-1,1]));
end

function test_create_boundingbox_3(dat)
% Combine test regions 1 and 3.
    bnd = dspace.combine_boundaries(dat.bnd,[1;3]);
    box = dspace.create_boundingbox(bnd);
    assertElementsAlmostEqual(box.x3,exp([-2,2]));
    assertElementsAlmostEqual(box.x4,exp([-1,1]));
end

function test_create_boundingbox_4(dat)
% Combine test regions 2 and 4.
    bnd = dspace.combine_boundaries(dat.bnd,[2;4]);
    box = dspace.create_boundingbox(bnd);
    assertElementsAlmostEqual(box.x3,exp([-1,1]));
    assertElementsAlmostEqual(box.x4,exp([-2,2]));
end

% Repeat for symbolic case.

function test_create_boundingbox_1_sym(dat)
% Combine test regions 3 and 4.
    bnd = dspace.combine_boundaries(dat.bndsym,[3;4]);
    box = dspace.create_boundingbox(bnd,dat.p0);
    assertElementsAlmostEqual(box.x3,exp([-1,1]));
    assertElementsAlmostEqual(box.x4,exp([-1,1]));
end

function test_create_boundingbox_2_sym(dat)
% Combine test regions 1 and 2.
    bnd = dspace.combine_boundaries(dat.bndsym,[1;2]);
    box = dspace.create_boundingbox(bnd,dat.p0);
    assertElementsAlmostEqual(box.x3,exp([-1,1]));
    assertElementsAlmostEqual(box.x4,exp([-1,1]));
end

function test_create_boundingbox_3_sym(dat)
% Combine test regions 1 and 3.
    bnd = dspace.combine_boundaries(dat.bndsym,[1;3]);
    box = dspace.create_boundingbox(bnd,dat.p0);
    assertElementsAlmostEqual(box.x3,exp([-2,2]));
    assertElementsAlmostEqual(box.x4,exp([-1,1]));
end

function test_create_boundingbox_4_sym(dat)
% Combine test regions 2 and 4.
    bnd = dspace.combine_boundaries(dat.bndsym,[2;4]);
    box = dspace.create_boundingbox(bnd,dat.p0);
    assertElementsAlmostEqual(box.x3,exp([-1,1]));
    assertElementsAlmostEqual(box.x4,exp([-2,2]));
end

% Show not implemented for incomplete symbolic case.

function test_create_boundingbox_err_sym(dat)
% Leave a variable in kinetic orders (nonlinear case).
    p0 = rmfield(dat.p0,'a1');
    bnd = dspace.combine_boundaries(dat.bndsym,[2;4]);
    f = @()dspace.create_boundingbox(bnd,p0);
    assertExceptionThrown(f,dspace.Settings.ERROR_NOT_IMPLEMENTED);
end
